# Schema System

LLMs need to know the structure you want. A schema tells the model "generate data in THIS format." But schemas cost tokens. 

## The Three Modes

```
╔══════════════════════════════════════════════════════════╗
║  .alwaysInclude                                          ║
╠══════════════════════════════════════════════════════════╣
║  Every prompt → [Your Prompt + Schema Description]       ║
║  Result: Most reliable, most tokens                      ║
╚══════════════════════════════════════════════════════════╝

╔══════════════════════════════════════════════════════════╗
║  .disabled                                               ║
╠══════════════════════════════════════════════════════════╣
║  Every prompt → [Your Prompt ONLY]                       ║
║  Result: Fewest tokens, relies on examples/memory        ║
╚══════════════════════════════════════════════════════════╝

╔══════════════════════════════════════════════════════════╗
║  .intelligent(hasExamples: Bool)                         ║
╠══════════════════════════════════════════════════════════╣
║  Decides dynamically: include schema or skip it          ║
║  Result: Smart optimization, balances tokens + quality   ║
╚══════════════════════════════════════════════════════════╝
```

## Intelligent Mode Logic

```
┏━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┓
┃ hasExamples = true?              ┃
┗━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━┛
         │
    ┌────┴────┐
   YES       NO
    │         │
    │         ▼
    │    ┏━━━━━━━━━━━━━━━━━━━━━━━━━━┓
    │    ┃ Type seen before?        ┃
    │    ┗━━━━━━━━━━━━━━━━━━━━━━━━━━┛
    │              │
    │         ┌────┴────┐
    │        YES       NO
    │         │         │
    ▼         ▼         ▼
[SKIP]    [SKIP]   [INCLUDE]
```

**Translation:**
- If you gave examples → skip schema (examples teach format)
- If no examples but type used before → skip schema (model remembers)
- If no examples and first time seeing type → include schema (needs it)

## Token Impact

```
5 requests, same type:

┌─────────────────────────────────────────┐
│ .alwaysInclude                          │
├─────────────────────────────────────────┤
│ [Prompt+Schema] [Prompt+Schema] ... ×5  │
│ = 5000 tokens                           │
└─────────────────────────────────────────┘

┌─────────────────────────────────────────┐
│ .intelligent (no examples)              │
├─────────────────────────────────────────┤
│ [Prompt+Schema] [Prompt] [Prompt] ... ×4│
│ = 3000 tokens (40% less)                │
└─────────────────────────────────────────┘

┌─────────────────────────────────────────┐
│ .intelligent (with examples)            │
├─────────────────────────────────────────┤
│ [Prompt] [Prompt] [Prompt] ... ×5       │
│ = 2500 tokens (50% less)                │
└─────────────────────────────────────────┘
```

## Behind The Scenes

### Types Used

```
SchemaHandling (enum)
  ├─ .intelligent(hasExamples: Bool)
  ├─ .alwaysInclude
  └─ .disabled

SchemaPolicy (struct)
  ├─ handling: SchemaHandling
  ├─ schemaIntelligence: SchemaIntelligence? (actor)
  └─ shouldIncludeSchema(forType:) async -> Bool

SchemaIntelligence (actor)
  ├─ hasExamples: Bool
  ├─ usedStructuredTypes: Set<String>
  └─ shouldIncludeSchema(forType:) -> Bool
```

### When You Create a Session

```
SessionConfiguration(
  schemaPolicy: SchemaPolicy(
    handling: .intelligent(hasExamples: true)
  )
)
    ↓
SchemaPolicy inits
    ↓
Spawns SchemaIntelligence actor (if intelligent mode)
    ↓
Actor tracks state across all requests
```

### During Generation

```
generate(prompt: "...", generating: MyType.self)
    ↓
Backend asks: policy.shouldIncludeSchema(forType: "MyType")
    ↓
Actor checks rules → returns true/false
    ↓
[true] → Attach schema to prompt
[false] → Send prompt without schema
    ↓
After success: policy.recordTypeUsage("MyType")
```

## Decision Guide

```
╔═══════════════════════════════════╦═════════════════════════╗
║ Scenario                          ║ Recommendation          ║
╠═══════════════════════════════════╬═════════════════════════╣
║ Provided examples in config       ║ .intelligent(true)      ║
║ Multi-turn, same type             ║ .intelligent(false)     ║
║ Maximum reliability needed        ║ .alwaysInclude          ║
║ Maximum token efficiency          ║ .disabled (risky)       ║
║ First time using a type           ║ .alwaysInclude or .intelligent ║
╚═══════════════════════════════════╩═════════════════════════╝
```

